/* retrieve type information from a type library or OLE class
 * Copyright 2001 Patrick TJ McPhee
 * Distributed under the terms of the Mozilla Public Licence
 * The original code is w32funcs
 * The original author is Patrick TJ McPhee
 */


#define STRICT
#include <windows.h>
#include <ole2ver.h>
#include <oleauto.h>
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>

#define INCL_REXXSAA
#define RX_STRONGTYPING
#include <rexxsaa.h>

#define V_GET 1         /* should update rexx variable on refresh */
#define V_SET 2         /* should update OLE variable on refresh */
#define V_OBJ 4         /* variable is an object, not a string */

/* array of properties to retrieve/update */
typedef struct olevar_T {
   BSTR doc;
   RXSTRING name;       /* name of variable (excluding stem name) */
   MEMBERID memid;      /* OLE member id */
   char * type;
   int flags;           /* flags for this variable */
} olevar_t;

/* constant value */
typedef struct oleconst_T {
   BSTR doc;
   RXSTRING name;       /* name of variable (excluding stem name) */
   int value;           /* value */
} oleconst_t;

/* array of functions which can be called, and arguments, etc */
typedef struct olefcn_T {
   BSTR doc;
   RXSTRING name;       /* name of function (excluding stem name) */
   MEMBERID memid;      /* OLE memer id */
   int args, oargs;     /* count of args and optional args */
   char * type;
   char **argt;       /* arg types -- we convert some scaler types, but classes are pointers */
   RXSTRING *argn;       /* arg names */
   unsigned short flags; /* invocation flags */
} olefcn_t;

/* a class definition. we build up a list of class defintions, and, hey-presto,
 * we've got ourselves a type lib. */

struct oleclass_t {
   oleclass_t *n;   /* next thing in list */
   BSTR doc;
   RXSTRING name;          /* name of type */
   olevar_t * va;          /* array of variables to update/retrieve */
   olefcn_t * fa;          /* array of functions */
   oleconst_t * ca;        /* array of constants */
   int vc, fc, cc;         /* counts of elements in va, fa, ca */
   oleclass_t(BSTR);
   ~oleclass_t();
   void dump(void);
};

static void bstrtorxstr(PRXSTRING rxs, BSTR s)
{
   if (!s) rxs->strlength = 0;
   else {
      rxs->strlength = wcslen(s);
      rxs->strptr = (char *)malloc(rxs->strlength);

      WideCharToMultiByte(CP_ACP, 0, s, rxs->strlength, rxs->strptr, rxs->strlength,
                          NULL, FALSE);
   }
}


oleclass_t::oleclass_t(BSTR uname)
{
   n = NULL;
   va = NULL;
   fa = NULL;
   ca = NULL;
   vc = fc = cc = 0;

   bstrtorxstr(&name, uname);
}

oleclass_t::~oleclass_t()
{
   if (name.strptr) free((void *)name.strptr);
   if (va) free((void *)va);
   if (fa) free((void *)fa);
   if (ca) free((void *)ca);
}

void oleclass_t::dump()
{
   register int i, j;

   printf("Type %.*s\n", name.strlength, name.strptr);

   if (cc) {
      printf("\tConstants:\n");
      for (i = 0; i < cc; i++) {
         printf("\t\t%.*s = %d", ca[i].name.strlength, ca[i].name.strptr,
                ca[i].value); 
         if (ca[i].doc) {
            printf("\t/* %ls */", ca[i].doc);
         }
         printf("\n");
      }
   }

   if (vc) {
      printf("\tProperties:\n");
      for (i = 0; i < vc; i++) {
         printf("\t\t%.*s (%s%s%s %s)", va[i].name.strlength, va[i].name.strptr, (va[i].flags & V_GET) ? "read" : "",
                (va[i].flags & V_SET) && (va[i].flags & V_GET) ? "/" : "",
                (va[i].flags & V_SET) ? "write" : "",
                va[i].type);
         if (va[i].doc) {
            printf("\t/* %ls */", va[i].doc);
         }
         printf("\n");
      }
   }

   if (fc) {
      static const char comma[] = ", ", optcomma[] = "[, ", nil[] = "",
                        optlpar[] = "[", optrpar[] = "]";

      printf("\tMethods:\n");
      for (i = 0; i < fc; i++) {
         if (fa[i].doc) {
            printf("\t\t/* %ls */\n", fa[i].doc);
         }

         printf("\t\t%.*s(%s", fa[i].name.strlength, fa[i].name.strptr, fa[i].args && fa[i].args == fa[i].oargs ? optlpar : nil);
         for (j = 0; j < fa[i].args; j++) {
            printf("%.*s %s%s", fa[i].argn[j].strlength, fa[i].argn[j].strptr,
                   fa[i].argt[j], j == (fa[i].args - 1) ? (fa[i].oargs ? optrpar : nil) : (fa[i].oargs - j - 1) ? comma : optcomma);
         }
        printf(") %s\n", fa[i].type);
      }
   }
}

static const char * vartype_aux(VARTYPE vt)
{
   static char dunnobuf[1024];
   const char * vartypes[] = {
    	"empty",
	"null",
	"number",
	"number",
	"number",
	"number",
	"number",
	"date",
	"string",
	"object",
	"error",
	"boolean",
	"many",
	"unknown",
	"number",
        "??",
	"number",
	"number",
	"number",
	"number",
	"number",
	"number",
	"number",
	"number",
	"no value",
	"error",
	"??",
	"??",
	"array",
	"??",
	"string",
	"string" },
     * vartypes64[] = {
	"time",
	"string",
	"??",
	"??",
	"??",
	"??",
	"??",
	"??",
	"classid"
     };

     if (vt >= VT_EMPTY && vt <= VT_LPWSTR)
        return vartypes[vt];
     else if (vt >= VT_FILETIME && vt <= VT_CLSID)
        return vartypes64[vt-VT_FILETIME];
     else if (vt == VT_VECTOR)
        return "VT_VECTOR";
     else if (vt == VT_ARRAY)
        return "VT_ARRAY";
     else if (vt == VT_BYREF)
        return "VT_BYREF";
     else if (vt == VT_RESERVED)
        return "VT_RESERVED";
     else if (vt == VT_ILLEGAL)
        return "VT_ILLEGAL";
     else if (vt == VT_ILLEGALMASKED)
        return "VT_ILLEGALMASKED";
     else if (vt == VT_TYPEMASK)
        return "VT_TYPEMASK";
     else {
        sprintf(dunnobuf, "dunno -- %x", vt);
        return dunnobuf;
     }
}

static VARTYPE true_vartype(TYPEDESC * tdesc)
{
   switch (tdesc->vt) {
      case VT_PTR:
      case VT_SAFEARRAY:
         return true_vartype(tdesc->lptdesc);
         break;

      default: return tdesc->vt; break;
   }
}

static HREFTYPE true_hreftype(TYPEDESC * tdesc)
{
   switch (tdesc->vt) {
      case VT_PTR:
      case VT_SAFEARRAY:
         return true_hreftype(tdesc->lptdesc);
         break;

      case VT_USERDEFINED:
         return tdesc->hreftype;
         break;


      default: return 0; break;
   }
}

static const char * vartype(LPTYPEINFO theTypeInfo, TYPEDESC * tdesc)
{
   static char dunnobuf[1024];
   LPTYPEINFO aTI = NULL;

   switch (tdesc->vt) {
      case VT_PTR:
      case VT_SAFEARRAY:
         return vartype(theTypeInfo, tdesc->lptdesc);
         break;

      case VT_USERDEFINED:
         if (theTypeInfo->GetRefTypeInfo(tdesc->hreftype, &aTI) == S_OK) {
            BSTR name;
            register int i;

            aTI->GetDocumentation(MEMBERID_NIL, &name, NULL, NULL, NULL);

            for (i = 0; name[i]; i++) {
               dunnobuf[i] = (char)name[i];
            }
            dunnobuf[i] = 0;
            SysFreeString(name);
            aTI->Release();
            return dunnobuf;
            break;
         }

         /* else flow through */
      default: return vartype_aux(tdesc->vt); break;
   }
}

static const char * varkind(VARKIND vk)
{
   const char * varkinds[] = {	"VAR_PERINSTANCE",
	"VAR_STATIC",
	"VAR_CONST",
	"VAR_DISPATCH",
    };

    if (vk >= 0 && vk <= VAR_DISPATCH) return varkinds[vk];
    else return "?";

}

static oleclass_t **dt = NULL, *classes = NULL;
static int typec = 0;

static int rxstrcmp(const PRXSTRING l, const PRXSTRING r)
{
   register int lt = l->strlength - r->strlength;

   if (lt)
      return lt;
   else
      return memcmp(l->strptr, r->strptr, l->strlength);
}

/* returns 1 if name is in dt, 0 otherwise. Sets index to the offset of
 * name if it's found, or the spot name should go if it isn't found */
static int tnamesearch(PRXSTRING name, int * index)
{
   register int r, l, i, c;
   
   for (c = 1, l = 0, r = typec - 1, i = r/2; i >= l && i <= r && c; i = (r - l)/2 + l) {
      c = rxstrcmp(name, &dt[i]->name);

      if (c < 0) {
         r = i - 1;
      }
      else if (c > 0) {
         l = i + 1;
      }
   }


   *index = i;
   return !c;
}


/* comparing guids didn't work too well */
static int didtype(oleclass_t * cl)
{
   int found = 0, index = 0;

   if (typec) {
      found = tnamesearch(&cl->name, &index);
   }

   if (found) return 1;
   else {
      if (!(typec % 1000)) {
         dt = (oleclass_t **)realloc(dt, sizeof(*dt)*(typec+1000));
      }

      if (index < typec)
         memmove((void *)(dt+index+1), (void *)(dt+index),
                    sizeof(*dt)*(typec-index));

      dt[index] = cl;
      typec++;
      return 0;
   }
}

static void cleantype()
{
   oleclass_t * cl1, *cl2;

   typec = 0;
   if (dt) free(dt);
   dt = NULL;

   for (cl1 = classes; cl1; cl1 = cl2) {
      cl2 = cl1->n;
      delete cl1;
   }

   classes = NULL;
}


static int gettypeinfo(LPTYPEINFO theTypeInfo)
{
   LPFUNCDESC theFunc;
   LPVARDESC theVar;
   LPTYPEATTR theTypeAttr;
   register int i,j, k;
   UINT count;
   HRESULT hr;
   RXSTRING nm;
   BSTR *names = (BSTR *)malloc(2*sizeof(*names));
   unsigned int cNames = 2;
   char * rbrack;
   ULONG helpctxt;
   int rc = 0;
   unsigned int cnt;
   oleclass_t * oc = NULL, *poc;

   /* find out some stuff about the type */
   if (theTypeInfo->GetTypeAttr(&theTypeAttr) != S_OK) {
      printf("oops\n");
      return -1;
   }

   /* make sure we're supposed to deal with this */
   if (
      /* sometimes we need to see private classes (e.g., word.application is
       * private! */
#ifdef NO_PRIVATE_CLASSES
   (theTypeAttr->wTypeFlags & (TYPEFLAG_FHIDDEN|TYPEFLAG_FRESTRICTED)) ||
#endif
       (theTypeAttr->typekind != TKIND_ENUM &&
        theTypeAttr->typekind != TKIND_DISPATCH &&
        theTypeAttr->typekind != TKIND_RECORD &&
        theTypeAttr->typekind != TKIND_COCLASS &&
        !(theTypeAttr->wTypeFlags & (TYPEFLAG_FDUAL)))) {
      theTypeInfo->ReleaseTypeAttr(theTypeAttr);
      return 1;
   }

   /* there's a chance we'll need this class, so we allocate it. We need an
    * oleclass to feed to didtype, mostly because I want to search with
    * RXSTRINGS rather than BSTRs */

   theTypeInfo->GetDocumentation(MEMBERID_NIL, names, names+1, NULL, NULL);

   oc = new oleclass_t(names[0]);

   if (didtype(oc)) {
      delete oc;
      SysFreeString(names[0]);
      theTypeInfo->ReleaseTypeAttr(theTypeAttr);
      if (names[1])
         SysFreeString(names[1]);
      return 2;
   }

   SysFreeString(names[0]);
   oc->doc = names[1];

   if (theTypeAttr->typekind == TKIND_COCLASS) {
      LPTYPEINFO pTI;
      HREFTYPE href;

      for (i = 0; i < theTypeAttr->cImplTypes; i++) {
         if (theTypeInfo->GetRefTypeOfImplType(i, &href) != S_OK ||
             theTypeInfo->GetRefTypeInfo(href, &pTI) != S_OK) {
            theTypeInfo->ReleaseTypeAttr(theTypeAttr);
            return -2;
         }
         gettypeinfo(pTI);
         pTI->Release();
      }

      oc->n = classes;
      classes = oc;

      return 0;
   }


   if (theTypeAttr->cVars) {
      oc->cc = 0;
      oc->ca = (oleconst_t *)malloc(sizeof(*oc->ca)*theTypeAttr->cVars);

      for (i = 0; i < theTypeAttr->cVars; i++) {
         if (theTypeInfo->GetVarDesc(i, &theVar) != S_OK) {
            printf("var oops\n");
            continue;
         }

         theTypeInfo->GetDocumentation(theVar->memid, names, names+1, NULL, NULL);

         bstrtorxstr(&nm, names[0]);
         SysFreeString(names[0]);

         if (theVar->varkind == VAR_CONST) {
            oc->ca[oc->cc].name = nm;
            oc->ca[oc->cc].value = theVar->lpvarValue->lVal;
            oc->ca[oc->cc++].doc = names[1];
         }
         else if (theVar->varkind == VAR_DISPATCH) {
            if (!(oc->vc % 20))
               oc->va = (olevar_t *)realloc(oc->va, sizeof(*oc->va)*(oc->vc+20));

            oc->va[oc->vc].name = nm;
            oc->va[oc->vc].memid = theVar->memid;
            oc->va[oc->vc].flags = V_GET;
            oc->va[oc->vc].type = strdup(vartype(theTypeInfo, &theVar->elemdescVar.tdesc));
            oc->va[oc->vc++].doc = names[1];
         }
         else {
            printf("%.*s is %s, not a constant\n", nm.strlength, nm.strptr, varkind(theVar->varkind));
         }

         theTypeInfo->ReleaseVarDesc(theVar);
      }
   }

   for (i = 0; i < theTypeAttr->cFuncs; i++) {
      if (theTypeInfo->GetFuncDesc(i, &theFunc) != S_OK) {
         printf("func oops\n");
         continue;
      }

      /* block off the functions the authors don't want you to know about,
       * but include _NewEnum */
      if (theFunc->funckind != FUNC_DISPATCH ||
          ((theFunc->wFuncFlags&(FUNCFLAG_FRESTRICTED|FUNCFLAG_FHIDDEN|FUNCFLAG_FNONBROWSABLE))
           && theFunc->memid != DISPID_NEWENUM)) {
         theTypeInfo->ReleaseFuncDesc(theFunc);
         continue;
      }

      /* treat as a function if it's a function or it has parameters. Treat
       * as a property if it's a property and it has no required members. */
      if ((theFunc->invkind & INVOKE_FUNC) ||
          ((theFunc->invkind & INVOKE_PROPERTYGET) && theFunc->cParams) ||
          ((theFunc->invkind & (INVOKE_PROPERTYPUT|INVOKE_PROPERTYPUTREF)) &&
           theFunc->cParams > 1)) {
         if (!(oc->fc % 20))
            oc->fa = (olefcn_t *)realloc(oc->fa, sizeof(*oc->fa)*(oc->fc+20));

         theTypeInfo->GetDocumentation(theFunc->memid, names, names+1, NULL, NULL);

         bstrtorxstr(&oc->fa[oc->fc].name, names[0]);
         oc->fa[oc->fc].doc = names[1];
         oc->fa[oc->fc].memid = theFunc->memid;
         oc->fa[oc->fc].args = theFunc->cParams;
         oc->fa[oc->fc].oargs = theFunc->cParamsOpt;
         oc->fa[oc->fc].type = strdup(vartype(theTypeInfo, &theFunc->elemdescFunc.tdesc));
         oc->fa[oc->fc].argn = (PRXSTRING)malloc(sizeof(RXSTRING)*theFunc->cParams);
         oc->fa[oc->fc].argt = (char **)malloc(sizeof(char *)*theFunc->cParams);
         oc->fa[oc->fc].flags = 0;

         if (cNames < (theFunc->cParams+1)) {
            cNames = theFunc->cParams + 1;
            names = (BSTR *)realloc(names, cNames*sizeof(*names));
         }

         theTypeInfo->GetNames(theFunc->memid, names, cNames, &cnt);

         for (j = 0; j < theFunc->cParams; j++) {
            k = j + 1;
            if (k < cnt) {
               bstrtorxstr(oc->fa[oc->fc].argn + j, names[k]);
               SysFreeString(names[k]);
            }
            else {
               bstrtorxstr(oc->fa[oc->fc].argn + j, names[0]);
            }
            oc->fa[oc->fc].argt[j] = strdup(vartype(theTypeInfo, &theFunc->lprgelemdescParam[j].tdesc));

            /* add a new typedef */
            if (true_vartype(&theFunc->lprgelemdescParam[j].tdesc) == VT_USERDEFINED) {
               LPTYPEINFO aTI;

               if (theTypeInfo->GetRefTypeInfo(true_hreftype(&theFunc->lprgelemdescParam[j].tdesc), &aTI)
                   == S_OK) {
                  gettypeinfo(aTI);
                  aTI->Release();
               }
            }
         }

         /* move the ,value' argument to the first element */
         if (cnt == theFunc->cParams) {
            char * argt = oc->fa[oc->fc].argt[j-1];
            RXSTRING argn = oc->fa[oc->fc].argn[j-1];
            memmove(oc->fa[oc->fc].argt+1, oc->fa[oc->fc].argt,
                    sizeof(*oc->fa[oc->fc].argt)*(theFunc->cParams - 1));
            memmove(oc->fa[oc->fc].argn+1, oc->fa[oc->fc].argn,
                    sizeof(*oc->fa[oc->fc].argn)*(theFunc->cParams - 1));
            oc->fa[oc->fc].argt[0] = argt;
            oc->fa[oc->fc].argn[0] = argn;
         }

         SysFreeString(names[0]);

         if (theFunc->invkind & (INVOKE_PROPERTYPUTREF|INVOKE_PROPERTYPUT))
            oc->fa[oc->fc].flags |= V_SET;
         
         if (theFunc->invkind & INVOKE_PROPERTYGET)
            oc->fa[oc->fc].flags |= V_GET;

         oc->fc++;
      }

      /* if it's a put or get and it doesn't require arguments, store it */
      if (((theFunc->invkind & INVOKE_PROPERTYGET) && theFunc->cParams == theFunc->cParamsOpt) ||
          ((theFunc->invkind & (INVOKE_PROPERTYPUT|INVOKE_PROPERTYPUTREF)) &&
           theFunc->cParams == theFunc->cParamsOpt+1)) {

         if (!(oc->vc % 20))
            oc->va = (olevar_t *)realloc(oc->va, sizeof(*oc->va)*(oc->vc+20));

         for (j = 0; j < oc->vc; j++) {
            if (oc->va[j].memid == theFunc->memid) {
               break;
            }
         }

         if (j == oc->vc) {
            oc->va[j].flags = 0;

            if (theFunc->invkind & (INVOKE_PROPERTYPUTREF|INVOKE_PROPERTYPUT))
               oc->va[oc->vc].type = strdup(vartype(theTypeInfo, &theFunc->lprgelemdescParam[0].tdesc));
            else
               oc->va[oc->vc].type = strdup(vartype(theTypeInfo, &theFunc->elemdescFunc.tdesc));


            theTypeInfo->GetDocumentation(theFunc->memid, names, names+1, NULL, NULL);
            bstrtorxstr(&oc->va[j].name, names[0]);
            SysFreeString(names[0]);
            oc->va[j].doc = names[1];
            oc->va[j].memid = theFunc->memid;
            oc->vc++;
         }

         if (theFunc->invkind & (INVOKE_PROPERTYPUTREF|INVOKE_PROPERTYPUT))
            oc->va[j].flags |= V_SET;
         else
            oc->va[j].flags |= V_GET;
      }

      /* add a new typedef */
      if (true_vartype(&theFunc->elemdescFunc.tdesc) == VT_USERDEFINED) {
         LPTYPEINFO aTI;

         if (theTypeInfo->GetRefTypeInfo(true_hreftype(&theFunc->elemdescFunc.tdesc), &aTI)
             == S_OK) {
            gettypeinfo(aTI);
            aTI->Release();
         }

      }
      theTypeInfo->ReleaseFuncDesc(theFunc);
   }

   theTypeInfo->ReleaseTypeAttr(theTypeAttr);

   oc->n = classes;
   classes = oc;


   return 0;
}

static int getdspinfo(LPDISPATCH theDispatch)
{
   LPTYPEINFO theTypeInfo = NULL;
   register int i;
   UINT count;
   HRESULT hr;
   oleclass_t * cl;
   int rc = 0;

   hr = theDispatch->GetTypeInfoCount(&count);

   for (i = 0; i < count; i++) {

      if ((hr = theDispatch->GetTypeInfo(i, LOCALE_USER_DEFAULT, &theTypeInfo)) != S_OK) {
         fprintf(stderr, "Error %x retrieving type info %d\n", hr, i);
         rc++;
      }

      gettypeinfo(theTypeInfo);
      theTypeInfo->Release();
   }

   for (cl = classes; cl; cl = cl->n) {
      cl->dump();
   }

   cleantype();

   return rc;
}


static int gettypeinfo(LPTYPELIB theTypeLib)
{
   LPTYPEINFO theTypeInfo = NULL;
   register int i;
   UINT count;
   HRESULT hr;
   oleclass_t * cl;
   int rc = 0;

   count = theTypeLib->GetTypeInfoCount();

   for (i = 0; i < count; i++) {
      if ((hr = theTypeLib->GetTypeInfo(i, &theTypeInfo)) != S_OK) {
         fprintf(stderr, "Error %x retrieving type info %d\n", hr, i);
         rc++;
         break;
      }

      gettypeinfo(theTypeInfo);
      theTypeInfo->Release();
   }

   for (cl = classes; cl; cl = cl->n) {
      cl->dump();
   }

   cleantype();

   return rc;
}


main(int ac, char **av)
{
   LPTYPELIB theTypeLib;
   LPDISPATCH theDispatch;
   LPUNKNOWN theUnknown;
   CLSID clsid;
   register int i;
   HRESULT hr;
   OLECHAR avone[FILENAME_MAX];
   int rc = 0;
   int depth = 1;

   if (ac < 2) {
      fprintf(stderr, "usage: %s progname\n", av[0]);
      return 1;
   }

   OleInitialize(NULL);

   for (i = 1; i < ac; i++) {

      /* we always need to convert the argument to unicode */
      MultiByteToWideChar(CP_ACP, 0, av[i], -1, avone, sizeof avone);

      /* assume the argument is a classid or program name */
      hr = CLSIDFromProgID(avone, &clsid);
      if (hr != S_OK)
         hr = CLSIDFromString(avone, &clsid);

      if (hr == S_OK) {
         printf("==\nclass %s\n", av[i]);

         try {
            hr = GetActiveObject(clsid, NULL, &theUnknown);
            if (hr != S_OK) {
               hr = CoCreateInstance(clsid, NULL, CLSCTX_SERVER, IID_IUnknown, (LPVOID *)&theUnknown);
            }
         }
         catch (...) {
            fprintf(stderr, "%s was very naughty and crashed during initialisation.\n", av[i]);
            rc++;
            continue;
         }

         if (hr == S_OK)
         {
            hr = theUnknown->QueryInterface(IID_IDispatch, (LPVOID *)&theDispatch);
            theUnknown->Release();

            if (hr == S_OK) {
               rc += getdspinfo(theDispatch);
               theDispatch->Release();
            }
            else {
               rc++;
               fprintf(stderr, "Error %x querying IDispatch interface for class %s.\n", hr, av[i]);
            }
         }
         else
         {
            rc++;
            fprintf(stderr, "Error %x loading class %s.\n", hr, av[i]); 
         }
      }

      else {
         /* perhaps it's a library, then */
         printf("==\nlibrary %s\n", av[i]);

         if ((hr = LoadTypeLib(avone, &theTypeLib)) != S_OK) {
            rc++;
            fprintf(stderr, "Error %x loading type library %s\n", hr, av[i]);
         }
         else {
            rc += gettypeinfo(theTypeLib);
            theTypeLib->Release();
         }
      }

   }

   return rc;
}

