#include <stdio.h>
/*
 *  Copyright 1995 Microsoft Corporation. All rights reserved.
 *  Developed by Ataman Software, Inc., ftp://rmii.com/pub2/ataman,
 *       info@ataman.com
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Modified by Patrick McPhee to be a loadable library, 1997/08/07
 * Changes copyright 1997, Patrick McPhee
 * $Header: C:/ptjm/rexx/w32funcs/RCS/w32funcs.c 1.25 2001/01/30 20:32:22 pmcphee Rel $
 */

#define STRICT
#include <windows.h>
#include <time.h>
#include <malloc.h>

#include "w32funcs.h"

static HKEY GetHKEY(PRXSTRING str);
static unsigned char *eLog, *theLog;

/* ataman code original raised a syntax error whenever anything went wrong,
 * for instance, if you tried to query a registry value which wasn't set.
 * I think you ought to be able to pass an invalid argument without it being
 * treated as a syntax error.  It does raise the question `how do we tell if
 * there's no value for a registry key?'  I return an empty string from from
 * the functions that return values, if there is no key matching the value.
 * If that's not good enough for you, I also set rc = 0 if the key existed,
 * and 1 if it didn't.  I'm not sure how usual it is to set rc from a
 * function, as opposed to an external command, but the ataman functions
 * don't really give a good way to figure it out.
 */

/* in the following lists of functions, I use the following conventions:
 *  key refers to a handle returned by one of the open functions, or
 *   to a key with a standard name, HKEY_LOCAL_MACHINE, HKEY_CLASSES_ROOT,
 *   HKEY_CURRENT_USER, or HKEY_USERS. These names can be in any case, and
 *   the `HKEY_' part is optional, since it's redundant.
 *  parameters in brackets are optional. parameters in braces are strings
 *   containing the name of a variable which will be used to return a value
 *  functions which return a key will return 0 to indicate failure.
 *   functions which return an arbitrary value will set rc if there's a
 *   problem of some sort
 *  functions which don't return any useful values return 0 for success and 1
 *   for failure
 *  these calls are meant to be upwardly compatible from the ataman calls
 *  but there is no warranty
 */

/* functions defined in this file:
 *  w32regopenkey(key, [subkeyname]) -> key
 *  w32regcreatekey(key, subkeyname) -> key
 *  w32regconnectregistry(hostname,key) -> key
 *  w32regclosekey(key) -> 0 (success) or 1 (failure)
 *  w32regqueryvalue(key, entry, [{type}], [{stem}]) -> the value (sets rc to 1 if no such entry)
 *  w32regqueryvaluetype(key, entry) -> type (use optional arg to w32regqueryvalue instead)
 *  w32regsetvalue(hkey, entry, type, data, [{stem}]) -> 0 or 1
 *  w32regdeletekey(hkey, subkeyname) -> 0 or 1
 *  w32regdeletevalue(hkey, valuename) -> 0 or 1
 *  w32regenumkey(hkey, [index], [{stem}]) -> keyname, 0 or 1 (sets rc)
 *  w32regenumvalue(hkey, [index], [{stem}]) -> value, 0 or 1 (sets rc)
 *  w32regflushkey(hkey) -> 0 or 1
 *  w32reggetkeysecdesc(hkey) -> secdesc (sets rc)
 *  w32regsetkeysecdesc(hkey, secdesc) -> 0 or 1
 *  w32regqueryinfokey(hkey, infokey, [{stem}]) -> desired info, 0 or 1 (sets rc)
 *  w32regsavekey(hkey, filename) -> 0 or 1
 *  w32regrestorekey(hkey, filename) -> 0 or 1
 *  w32regloadkey(hkey, subkey, filename) -> 0 or 1
 *  w32regunloadkey(hkey, subkey) -> 0 or 1
 *
 *  w32openeventlog(host, source) -> handle, or 0
 *  w32closeeventlog(handle) -> 0 or 1
 *  w32getnumberofeventlogrecords(handle) -> the number, or -1
 *  w32openbackupeventlog([hostname], filename) -> handle, or 0
 *  w32backupeventlog(handle, filename) -> 0 or 1
 *  w32backupeventlog(handle, [filename]) -> 0 or 1
 *  w32findeventlogentry(handle, recnum, [{stem}]) -> 0 or 1
 *  w32geteventid() -> id (sets rc)
 *  w32geteventtype() -> type (sets rc)
 *  w32geteventcategory() -> category (sets rc)
 *  w32geteventnumstrings() -> num strings (sets rc)
 *  w32geteventtimewritten() -> time written (sets rc)
 *  w32geteventtimegenerated() -> time generated (sets rc)
 *  w32geteventstring(index) -> string (sets rc)
 *  w32geteventdata() -> data (sets rc)
 *  w32writeeventlog([hostname], source, [eventtype], [category], [eventid], [data], [string1, ...]) -> 0 or 1
 *  w32writeevent({stem}) -> 0 or 1
 */

static char * type_name(DWORD dwType);
static DWORD type_type(const char * const theType);

/* open a registry key.
 *  arg1: the parent key (an hkey or one of the standard names)
 *  arg2: the key to open
 */
rxfunc(w32regopenkey)
{
	char *subkeyname = NULL;
	HKEY hkey, hkeyRet;
	DWORD err;
	int rc = 0;

	checkparam(1, 2) ;

	hkey = GetHKEY(argv);

	if (argc > 1) {
		rxstrdup(subkeyname, argv[1]);
	}

	if ((err=RegOpenKeyEx(hkey, subkeyname, 0, MAXIMUM_ALLOWED, &hkeyRet)) != ERROR_SUCCESS) {
            result_zero();
	}
	else {
	    result->strlength = sprintf(result->strptr, "%lx", (unsigned long)hkeyRet);
	}

	return rc;
}


/* hkey = w32_regcreatekey(hkey, subkeyname) */
rxfunc(w32regcreatekey)
{
	char *subkeyname = NULL;
	HKEY hkey, hkeyRet;
	DWORD err, disp;

	checkparam(2, 2) ;

	hkey = GetHKEY(argv);

	rxstrdup(subkeyname, argv[1]);

	if ((err=RegCreateKeyEx(hkey, subkeyname, 0, "", REG_OPTION_NON_VOLATILE,
		MAXIMUM_ALLOWED, NULL, &hkeyRet, &disp)) != ERROR_SUCCESS) {
           result_zero();
	}
	else {
	    result->strlength = sprintf(result->strptr, "%lx", (unsigned long)hkeyRet);
	}

	return 0;
}


rxfunc(w32regconnectregistry)
{
	/* hkey = w32_regconnectregistry(hostname, hkey) */
	char *hostname = NULL;
	HKEY hkey, hkeyRet;
	DWORD err;

	checkparam(2, 2) ;

	rxstrdup(hostname, argv[0]);

	hkey = GetHKEY(argv+1);

	if ((err=RegConnectRegistry(hostname, hkey, &hkeyRet)) != ERROR_SUCCESS) {
            result_zero();
	}
        else {
           result->strlength = sprintf(result->strptr, "%lx", (unsigned long)hkeyRet);
        }
	return 0;
}


rxfunc(w32regclosekey)
{
	/* call w32RegCloseKey hkey */
	HKEY hkey;
	char * key;

	checkparam(1, 1) ;

	rxstrdup(key, argv[0]);

	sscanf(key, "%lx", (unsigned long *)&hkey);

	if (RegCloseKey(hkey) != ERROR_SUCCESS) {
	    result_one();
	}
	else {
	    result_zero();
	}

	return 0;
}



rxfunc(w32regqueryvalue)
{
	char *valuename = NULL;
	HKEY hkey;
	DWORD cbData, dwType;
	unsigned char *byData = NULL;
	register int i;

	checkparam(2, 4) ;

	hkey = GetHKEY(argv);

	rxstrdup(valuename, argv[1]);

	result->strlength = 0;
	if (RegQueryValueEx(hkey, valuename, NULL, &dwType, NULL, &cbData) != ERROR_SUCCESS) {
		rc_one();
	}
	else {
	    byData = alloca(cbData);

	    if (!byData) {
		    rc_one();
		    return 0;
	    }

	    if (RegQueryValueEx(hkey, valuename, NULL, &dwType, byData, &cbData) != ERROR_SUCCESS) {
		    rc_one();
		    return 0;
	    }

	    if (argc > 2)
		setavar(argv+2, type_name(dwType), -1);

	    switch(dwType) {
	    case REG_DWORD_BIG_ENDIAN:
                    /* convert to little-endian and treat as a DWORD */
		    {
		    char holder[4];
                    holder[0] = byData[3];
                    holder[1] = byData[2];
                    holder[2] = byData[1];
                    holder[3] = byData[0];
                    memcpy(byData, holder, 4);
		    }
                    /* flow through */
	    case REG_DWORD:
		    {
		    result->strlength = sprintf(result->strptr, "%ld", *((long *)byData));
		    break;
		    }
	    case REG_EXPAND_SZ:
	    case REG_SZ:
		    {
		    /* make sure there's enough space. the string in byData
		     * includes a terminating null which we don't want */
                    if (cbData) {
                       rxresize(result, cbData - 1);
                       memcpy(result->strptr, byData, result->strlength);
                    }
                    else
                       result->strlength = 0;
		    break;
		    }
            /* if a stem variable name was passed, put the data in it */
	    case REG_MULTI_SZ:
	       if (argc >= 4) {
                  chararray *vals = new_chararray();
                  int l, offs = 0;

                  /* figure out how many there are */
                  for (offs = 0; offs < (cbData-1); offs += l + 1) {
                     l = strlen(byData+offs);
                     addstr(vals, byData+offs, l);
		  }
                  setastem(argv+3, vals);
                  delete_chararray(vals);
               }
               /* flow through, for backward compatibility */
	    default:	/* some kind of binary data */
		    {
		    rxresize(result, cbData*2+1);

		    result->strlength = 0;
		    for (i = 0; i < cbData; i++) {
			    result->strlength += sprintf(result->strptr+result->strlength, "%02x", (unsigned) byData[i]);
		    }
		    break;
		    }
	    }
	    rc_zero();
	}

	return 0;
}


/* return just the type of the registry entry. Since you almost never
 * want to do this without also querying the value of the entry, I added
 * an extra (optional) argument argument to w32regqueryvalue, which
 * holds the type. I leave this function for compatibility */
rxfunc(w32regqueryvaluetype)
{
	char *valuename = NULL;
	HKEY hkey;
	DWORD dwType;

	checkparam(2, 2) ;

	hkey = GetHKEY(argv);

	rxstrdup(valuename, argv[1]);

	if (RegQueryValueEx(hkey, valuename, NULL, &dwType, NULL, NULL) != ERROR_SUCCESS) {
		rc_one();
	}
	else {
	    rc_zero();
	    strcpy(result->strptr, type_name(dwType));
	    result->strlength = strlen(result->strptr);
	}

	return 0;
}



rxfunc(w32regsetvalue)
{
   char *valuename = NULL;
   char *type = NULL;
   char *data = NULL;
   HKEY hkey;
   DWORD dwType, cbData;
   unsigned char *byData = NULL;

   checkparam(4, 5) ;

   hkey = GetHKEY(argv);

   rxstrdup(valuename, argv[1]);
   rxstrdup(type, argv[2]);
   rxstrdup(data, argv[3]);

   dwType = type_type(type);

   /* assume success */
   result_zero();

   switch(dwType) {
       case REG_DWORD:
       case REG_DWORD_BIG_ENDIAN: {
          byData = alloca(sizeof (DWORD));
          if (!byData) {
             result_one();
          }
          else {
             sscanf(data, "%ld", (long *)byData);
             cbData = sizeof (DWORD);
          }
          if (dwType == REG_DWORD_BIG_ENDIAN) {
             char holder[4];
             holder[0] = byData[3];
             holder[1] = byData[2];
             holder[2] = byData[1];
             holder[3] = byData[0];
             memcpy(byData, holder, 4);
          }
          break;
       }

    case REG_EXPAND_SZ:
    case REG_SZ: {
       cbData = argv[3].strlength+1;
       byData = data;
       break;
    }

    case 0: result_one(); break;

    /* if we were passed the optional stem argument, build a bunch of concatentated
     * strings from it */
    case REG_MULTI_SZ: if (argc >= 5 && argv[4].strlength) {
       chararray * vals = new_chararray();
       register int i, j;

       getastem(argv+4, vals);


       /* the format we need happens to be the same as the chararray's
        * storage format, but we need an extra null at the end */
       cbData = vals->char_used + 1;
       byData = alloca(cbData);

       if (!byData) {
          result_one();
       }
       else {
          memcpy(byData, vals->chardata, vals->char_used);
          byData[vals->char_used] = 0;
       }
       delete_chararray(vals);

       break;
    }
    /* otherwise, flow through */

   default: {	/* some kind of binary data */
      DWORD dw;
      char *p;

      cbData = argv[3].strlength;
      if ((cbData%2) != 0) {
         result_one();
      }
      else {
         cbData /= 2;

         byData = alloca(cbData);
         if (!byData) {
            result_one();
         }
         else
            for (p=data, dw=0; dw<cbData; dw++, p+=2) {
               char buf[3];
               unsigned int x;
               buf[0] = *p;
               buf[1] = *(p+1);
               buf[2] = '\0';
               if (sscanf(buf, "%02x", &x) != 1) {
                  result_one();
               }
               byData[dw] = (unsigned char)x;
            }
         }
         break;
      }
   }

   /* do the update unless I've called the result_one() macro along the way */
   if (result->strptr[0] == '0')
      if (RegSetValueEx(hkey, valuename, 0, dwType, byData, cbData) != ERROR_SUCCESS) {
         result_one();
      }

   return 0;
}


/* w32expandenvironmentstrings is dropped because the functionality is
 * provided by, eg, value('PATH',,'ENVIRONMENT');
 */

static int delete_keys_recursively(HKEY hkey, const char * subkeyname)
{
   HKEY subhkey;
   int rc;
   register int i;
   char keybuf[MAX_PATH+1];
   
   rc = RegOpenKeyEx(hkey, subkeyname, 0, MAXIMUM_ALLOWED, &subhkey);

   /* if we opened it, recursively delete all the sub-keys, then get
    * rid of this one */
   if (rc == ERROR_SUCCESS) {
      for (i = 0, rc = RegEnumKey(subhkey, i, keybuf, sizeof(keybuf));
           rc == ERROR_SUCCESS;
           i++, rc = RegEnumKey(subhkey, i, keybuf, sizeof(keybuf))) {
         /* if this fails, I don't really care why. Maybe I should, but I
          * don't */
         delete_keys_recursively(subhkey, keybuf);
      }

      RegCloseKey(subhkey);
      rc = RegDeleteKey(hkey, subkeyname);
   }

   return rc;
}


rxfunc(w32regdeletekey)
{
   char *subkeyname = NULL;
   HKEY hkey;
   DWORD err;

   checkparam(2, 2) ;

   hkey = GetHKEY(argv);

   rxstrdup(subkeyname, argv[1]);

   err = RegDeleteKey(hkey, subkeyname);

   /* NT does not delete keys which contain sub-keys. WIN95/98
    * does, so I guess it's not by design...I want to delete
    * them in either case */
   if (GetLastError() == ERROR_PROC_NOT_FOUND) {
      /* I know that says `Error Proc not found', and the text that
       * goes with it says that a process was not found, but the
       * real problem is that you can't delete keys recursively. */
      err = delete_keys_recursively(hkey, subkeyname);
   }

   if (err != ERROR_SUCCESS)
      result_one();
   else
      result_zero();

   return 0;
}


rxfunc(w32regdeletevalue)
{
	char *valuename = NULL;
	HKEY hkey;

	checkparam(2, 2) ;

	hkey = GetHKEY(argv);

	rxstrdup(valuename, argv[1]);

	if (RegDeleteValue(hkey, valuename) != ERROR_SUCCESS) {
            result_one();
	}
	else {
		result_zero();
	}

	return 0;
}


/* regenumkey either returns the name of the ith key under key,
 * (original syntax), or it gets all the key names and sticks them in
 * a stem variable (my syntax). If you specify the index argument,
 * it returns an empty string if there is no such key. If you don't
 * it returns 0 if it could get all the keys, or 1 otherwise.
 * It's an error to specify neither the index nor the stem name */
rxfunc(w32regenumkey)
{
	HKEY hkey;
	DWORD index;
	char keybuf[MAX_PATH+1];
	char * indptr;

	/* you must specify at least one of index or stem */
	if (argc == 1 || argc > 3 || (argc == 2 && argv[1].strptr == NULL) ||
	    (argc == 3 && argv[2].strptr == NULL && argv[1].strptr == NULL))
	    return BADARGS;

	hkey = GetHKEY(argv);

	if (!hkey) {
	    if (!argv[1].strptr)
		result_one();
	    rc_one();
	    return 0;
	}

	if (argv[1].strptr) {
	    rxstrdup(indptr, argv[1]);
	    if (sscanf(indptr, "%lu", &index) != 1) {
		rc_one();
	    }

	    if (RegEnumKey(hkey, index, keybuf, sizeof keybuf) != ERROR_SUCCESS) {
		rc_one();
	    }
	    else {
		rc_zero();
		rxresize(result, strlen(keybuf));
		memcpy(result->strptr, keybuf, result->strlength);
	    }
	}

	/* if we specified a stem variable, we need to loop through the
	 * keys and set up a stem variable. The variable name is assumed
	 * to end in a `.' */
	if (argc == 3) {
            chararray *keys = new_chararray();
	    register int i;
	    int rc = ERROR_SUCCESS;

	    /* get all the keys */
	    for (i = 0, rc = RegEnumKey(hkey, i, keybuf, sizeof(keybuf));
		 rc == ERROR_SUCCESS;
		 i++, rc = RegEnumKey(hkey, i, keybuf, sizeof(keybuf))) {
                addstr(keys, keybuf, strlen(keybuf));
	    }

            if (rc == ERROR_NO_MORE_ITEMS)
               rc_zero();
            else {
               rc_one();
            }

	    /* now set the stem */
	    setastem(argv+2, keys);

	    /* clean up */
            delete_chararray(keys);

	    /* if we weren't returning a value, set result to 0 */
	    if (!argv[1].strptr) {
                if (rc == ERROR_NO_MORE_ITEMS)
                   result_zero();
                else
                   result_one();
	    }
	}

	return 0;
}


/* this behaves similarly to w32regenumkey, but for entry names. */
rxfunc(w32regenumvalue)
{
	HKEY hkey;
	DWORD index;
	char byNameBuf[MAX_PATH+1];
	DWORD cbNameBuf;
	char * indptr;

	/* you must specify at least one of index or stem */
	if (argc == 1 || argc > 3 || (argc == 2 && argv[1].strptr == NULL) ||
	    (argc == 3 && argv[2].strptr == NULL && argv[1].strptr == NULL))
	    return BADARGS;

	hkey = GetHKEY(argv);

	if (!hkey) {
	    if (!argv[1].strptr)
		result_one();
	    rc_one();
	    return 0;
	}

	if (argv[1].strptr) {
	    rxstrdup(indptr, argv[1]);
	    if (sscanf(indptr, "%lu", &index) != 1) {
		rc_one();
	    }

	    cbNameBuf = sizeof(byNameBuf);
	    /* there's probably all kinds of useful information being thrown
	     * away here */
	    if (RegEnumValue(hkey, index, byNameBuf, &cbNameBuf, NULL, NULL, NULL, NULL) != ERROR_SUCCESS) {
		rc_one();
	    }
	    else {
		rc_zero();
		rxresize(result, strlen(byNameBuf));
		memcpy(result->strptr, byNameBuf, result->strlength);
	    }
	}

	/* if we specified a stem variable, we need to loop through the
	 * values and set up a stem variable. The variable name is assumed
	 * to end in a `.' */
	if (argc == 3) {
	    chararray *vals = new_chararray();
	    register int i;
	    int rc = ERROR_SUCCESS;

	    /* get all the values */
	    for (i = 0, cbNameBuf = sizeof(byNameBuf), rc = RegEnumValue(hkey, i, byNameBuf, &cbNameBuf, NULL, NULL, NULL, NULL);
		 rc == ERROR_SUCCESS;
		 i++, cbNameBuf = sizeof(byNameBuf), rc = RegEnumValue(hkey, i, byNameBuf, &cbNameBuf, NULL, NULL, NULL, NULL)) {

		addstr(vals, byNameBuf, cbNameBuf);
	    }

            if (rc == ERROR_NO_MORE_ITEMS)
               rc_zero();
            else {
               rc_one();
            }

	    /* now set the stem */
	    setastem(argv+2, vals);

	    /* clean up */
	    delete_chararray(vals);

	    /* if we weren't returning a value, set result to 0 */
	    if (!argv[1].strptr) {
                if (rc == ERROR_NO_MORE_ITEMS)
                   result_zero();
                else
                   result_one();
	    }
	}

	return 0;
}


rxfunc(w32regflushkey)
{
	HKEY hkey;

	checkparam(1, 1) ;

	hkey = GetHKEY(argv);

	if (RegFlushKey(hkey) != ERROR_SUCCESS) {
	    result_one();
	}
	else {
	    result_zero();
	}

	return 0;
}


rxfunc(w32reggetkeysecdesc)
{
	HKEY hkey;
	PSECURITY_DESCRIPTOR psd = NULL;
	char *szbuf;
	DWORD cbsd, dw, err;
	SECURITY_INFORMATION si = OWNER_SECURITY_INFORMATION |
		GROUP_SECURITY_INFORMATION | DACL_SECURITY_INFORMATION;

	checkparam(1, 1) ;

	hkey = GetHKEY(argv);

	cbsd = 0;
	result->strlength = 0;
	if ((err=RegGetKeySecurity(hkey, si, NULL, &cbsd)) != ERROR_INSUFFICIENT_BUFFER) {
		rc_one();
		return 0;
	}

	psd = malloc(cbsd);
	if (!psd) {
		goto skipout;
	}

	szbuf = malloc((cbsd*2)+1);
	if (!szbuf) {
		goto skipout;
	}

	if (RegGetKeySecurity(hkey, si, psd, &cbsd) != ERROR_SUCCESS) {
		rc_one();
		goto skipout;
	}

	for (dw = 0; dw < cbsd; dw++) {
		sprintf(&szbuf[dw*2], "%02x", (unsigned) ((unsigned char *)psd)[dw]);
	}

	rxresize(result, cbsd*2);
	memcpy(result->strptr, szbuf, result->strlength);

	rc_zero();
skipout:
	if (psd) {
		free(psd);
	}

	if (szbuf) {
		free(szbuf);
	}

	return 0;
}



rxfunc(w32regsetkeysecdesc)
{
	char *secdesc = NULL;
	HKEY hkey;
	PSECURITY_DESCRIPTOR psd;
	DWORD cbpsd;
	DWORD dw;
	char *p;
	SECURITY_INFORMATION si = OWNER_SECURITY_INFORMATION |
		GROUP_SECURITY_INFORMATION | DACL_SECURITY_INFORMATION;

	checkparam(2, 2) ;

	hkey = GetHKEY(argv);

	rxstrdup(secdesc, argv[1]);

	cbpsd = strlen(secdesc);
	if ((cbpsd%2) != 0) {
		result_one();
		goto skipout;
	}
	cbpsd /= 2;

	psd = malloc(cbpsd);
	if (!psd) {
		result_one();
		goto skipout;
	}
	for (p=secdesc, dw=0; dw<cbpsd; dw++, p+=2) {
		char buf[3];
		unsigned int x;
		buf[0] = *p;
		buf[1] = *(p+1);
		buf[2] = '\0';
		if (sscanf(buf, "%02x", (unsigned) &x) != 1) {
			result_one();
			goto skipout;
		}
		((unsigned char *)psd)[dw] = (unsigned char)x;
	}

	if (RegSetKeySecurity(hkey, si, psd) == ERROR_SUCCESS) {
            result_zero();
	}

skipout:
	if (psd) {
		free(psd);
	}

	return 0;
}


rxfunc(w32regqueryinfokey)
{
	HKEY hkey;
	char *infoname;
	char numbuf[20];
	DWORD dwRet, dwDummy, err;
	FILETIME ftDummy;
	char chDummy[MAX_PATH+1];
	static const char * infothing[] = {
	    "NumSubKeys","MaxSubKeyName","NumValues","MaxValueName",
	    "MaxValueData"
	};
	DWORD infovalue[DIM(infothing)];
	register int i;

	checkparam(2, 3) ;

	hkey = GetHKEY(argv);

	rxstrdup(infoname, argv[1]);


	dwDummy = sizeof chDummy;
	if ((err=RegQueryInfoKey(hkey, chDummy, &dwDummy,
		NULL, infovalue, infovalue+1, &dwDummy,
		infovalue+2, infovalue+3, infovalue+4, &dwDummy, &ftDummy)) != ERROR_SUCCESS) {
		rc_one();
	}
	else {
		rc_zero();
	}

	/* default rc is -1 */
	dwRet = -1;
	for (i = 0; i < DIM(infothing); i++)
	    if (stricmp(infoname, infothing[i]) == 0) {
		dwRet = infovalue[i];
	    }

	result->strlength = sprintf(result->strptr, "%lu", dwRet);

	/* if we were passed a stem name, set a bunch of stem variables */
	if (argc == 3) {
	    SHVBLOCK svh[DIM(infothing)];
	    register int i;
	    char * name;

	    rxstrdup(name, argv[3]);

	    for (i = 0;  i < DIM(infothing); i++) {
		svh[i].shvnext = svh+i+1;
		svh[i].shvname.strptr = alloca(argv[2].strlength+15);
		svh[i].shvname.strlength = sprintf(svh[i].shvname.strptr, "%s%d", name, infothing[i]);
		svh[i].shvvalue.strptr = alloca(15);
		svh[i].shvvalue.strlength = sprintf(svh[i].shvvalue.strptr, "%d", infovalue[i]);
		svh[i].shvcode = RXSHV_SET;
	    }
	    svh[DIM(infothing)-1].shvnext = NULL;
	    RexxVariablePool(svh);
	}

	return 0;
}

const char se_backup_name[] = SE_BACKUP_NAME;
const char se_restore_name[] = SE_RESTORE_NAME;

static APIRET w32regsaverestorekey(ULONG argc, PRXSTRING argv, const char * saveparm, PRXSTRING result)
{


    char *filename;
    HKEY hkey;
    DWORD err;
    HANDLE ht;
    LUID luidValue;
    TOKEN_PRIVILEGES tkp;

    checkparam(2, 2) ;

    hkey = GetHKEY(argv);

    rxstrdup(filename, argv[1]);

    if (!OpenProcessToken(GetCurrentProcess(),
			  TOKEN_ADJUST_PRIVILEGES|TOKEN_QUERY, &ht) ||
	!LookupPrivilegeValue(NULL, saveparm, &luidValue)) {
	result_one();
    }

    else {
	tkp.PrivilegeCount = 1;
	tkp.Privileges[0].Luid = luidValue;
	tkp.Privileges[0].Attributes = SE_PRIVILEGE_ENABLED;

	if (!AdjustTokenPrivileges(ht, FALSE, &tkp, sizeof tkp, NULL, NULL) ||
	   (saveparm != se_backup_name && saveparm != se_restore_name) ||
	   (saveparm == se_backup_name && ((err=RegSaveKey(hkey, filename, NULL)) != ERROR_SUCCESS)) ||
	   (saveparm == se_restore_name && ((err=RegRestoreKey(hkey, filename, 0)) != ERROR_SUCCESS))) {
	    result_one();
	}
	else {
	    result_zero();
	}

	tkp.Privileges[0].Attributes = 0;

	if(!AdjustTokenPrivileges(ht, FALSE, &tkp, sizeof tkp, NULL, NULL)) {
	    result_one();
	}
    }

    return 0;
}


rxfunc(w32regsavekey)
{
    return w32regsaverestorekey(argc, argv, se_backup_name, result);
}

rxfunc(w32regrestorekey)
{
    return w32regsaverestorekey(argc, argv, se_restore_name, result);
}

static APIRET w32regloadunload(int nparms, ULONG argc, PRXSTRING argv, const char * saveparm, PRXSTRING result)
{
    char *filename;
    char *subkeyname;
    HKEY hkey;
    DWORD err;
    HANDLE ht;
    LUID luidValue;
    TOKEN_PRIVILEGES tkp;

    checkparam(nparms, nparms) ;

    hkey = GetHKEY(argv);

    rxstrdup(subkeyname, argv[1]);
    if (argc == 3)
	rxstrdup(filename, argv[2]);

    if (!OpenProcessToken(GetCurrentProcess(),
			  TOKEN_ADJUST_PRIVILEGES|TOKEN_QUERY, &ht) ||
	!LookupPrivilegeValue(NULL, saveparm, &luidValue)) {
	result_one();
    }

    else {
	tkp.PrivilegeCount = 1;
	tkp.Privileges[0].Luid = luidValue;
	tkp.Privileges[0].Attributes = SE_PRIVILEGE_ENABLED;

	if (!AdjustTokenPrivileges(ht, FALSE, &tkp, sizeof tkp, NULL, NULL) ||
	   (saveparm != se_backup_name && saveparm != se_restore_name) ||
	   (saveparm == se_backup_name && ((err=RegLoadKey(hkey, subkeyname, filename)) != ERROR_SUCCESS)) ||
	   (saveparm == se_restore_name && ((err=RegUnLoadKey(hkey, subkeyname)) != ERROR_SUCCESS))) {
          result_one();
	}
	else {
	    result_zero();
	}

	tkp.Privileges[0].Attributes = 0;

	if(!AdjustTokenPrivileges(ht, FALSE, &tkp, sizeof tkp, NULL, NULL)) {
	    result_one();
	}
    }

    return 0;
}

rxfunc(w32regloadkey)
{
    return w32regloadunload(3, argc, argv, se_backup_name, result);
}

rxfunc(w32regunloadkey)
{
    return w32regloadunload(2, argc, argv, se_restore_name, result);
}


static struct hkey_map_s {
    char * name;
    HKEY hkey;
}  stdkeynames[] = {
    "CLASSES_ROOT", HKEY_CLASSES_ROOT,
    "CURRENT_CONFIG", HKEY_CURRENT_CONFIG,
    "CURRENT_USER", HKEY_CURRENT_USER,
    "DYN_DATA", HKEY_DYN_DATA,
    "HKEY_CLASSES_ROOT", HKEY_CLASSES_ROOT,
    "HKEY_CURRENT_CONFIG", HKEY_CURRENT_CONFIG,
    "HKEY_CURRENT_USER", HKEY_CURRENT_USER,
    "HKEY_DYN_DATA", HKEY_DYN_DATA,
    "HKEY_LOCAL_MACHINE", HKEY_LOCAL_MACHINE,
    "HKEY_PERFORMANCE_DATA", HKEY_PERFORMANCE_DATA,
    "HKEY_USERS", HKEY_USERS,
    "LOCAL_MACHINE", HKEY_LOCAL_MACHINE,
    "PERFORMANCE_DATA", HKEY_PERFORMANCE_DATA,
    "USERS", HKEY_USERS,
};

/* key must be one of the bits of text listed here, or some hex value */
static HKEY GetHKEY(PRXSTRING str) {
	HKEY hkey = 0;
	register int i;

	for (i = 0; i < DIM(stdkeynames); i++)
	    if (!strnicmp(str->strptr, stdkeynames[i].name, str->strlength)) {
		hkey = stdkeynames[i].hkey;
		break;
	    }

	if (!hkey) {
	    if (sscanf(str->strptr, "%lx", &hkey) != 1) {
		hkey = 0;
	    }
	}

	return hkey;
}


rxfunc(w32openeventlog)
{
    char *hostname = NULL;
    char *sourcename;
    HANDLE hRet;

    /* must have two arguments, although 1 is optional */
    if (argc != 2 || !argv[1].strptr) {
	return BADARGS;
    }

    if (argv[0].strptr) {
	rxstrdup(hostname, argv[0]);
    }

    rxstrdup(sourcename, argv[1]);
    theLog = strdup(sourcename);

    if ((hRet=OpenEventLog(hostname, sourcename)) == NULL) {
	result_zero();
    }
    else {
	result->strlength = sprintf(result->strptr, "%lx", (unsigned long)hRet);
    }

    return 0;
}


rxfunc(w32closeeventlog)
{
    HANDLE h;
    char * hptr;

    checkparam(1, 1) ;

    rxstrdup(hptr, argv[0]);

    sscanf(hptr, "%lx", (unsigned long *)&h);

    if (!CloseEventLog(h)) {
	result_one();
    }
    else {
	result_zero();
    }

    if (eLog) {
	free(eLog);
	eLog = NULL;
    }

    return 0;
}



rxfunc(w32getnumberofeventlogrecords)
{
    HANDLE h;
    DWORD num;
    char * hptr;

    checkparam(1, 1) ;

    rxstrdup(hptr, argv[0]);

    sscanf(hptr, "%lx", (unsigned long *)&h);

    if (!GetNumberOfEventLogRecords(h, &num)) {
	num = -1;
    }

    result->strlength = sprintf(result->strptr, "%d", num);
    return 0;
}

rxfunc(w32getoldesteventlogrecord)
{
    HANDLE h;
    DWORD num;
    char * hptr;

    checkparam(1, 1) ;

    rxstrdup(hptr, argv[0]);

    sscanf(hptr, "%lx", (unsigned long *)&h);

    if (!GetOldestEventLogRecord(h, &num)) {
	num = -1;
    }

    result->strlength = sprintf(result->strptr, "%d", num);
    return 0;
}

rxfunc(w32openbackupeventlog)
{
    char *hostname = NULL;
    char *filename;
    HANDLE hRet;

    if (argc != 2 || !argv[1].strptr) {
	return BADARGS;
    }

    if (argv[0].strptr) {
	rxstrdup(hostname, argv[0]);
    }

    rxstrdup(filename, argv[1]);

    if ((hRet=OpenBackupEventLog(hostname, filename)) == NULL) {
	result_zero();
    }

    else {
	result->strlength = sprintf(result->strptr, "%lx", (unsigned long)hRet);
    }

    return 0;
}


rxfunc(w32backupeventlog)
{
    char *filename;
    HANDLE h;
    char * hptr;

    checkparam(2, 2) ;

    rxstrdup(hptr, argv[0]);
    sscanf(hptr, "%lx", (unsigned long *)&h);

    rxstrdup(filename, argv[1]);

    if (!BackupEventLog(h, filename)) {
    	result_one();
    }
    else {
	result_zero();
    }

    return 0;
}


rxfunc(w32cleareventlog)
{
    char *filename = NULL;
    HANDLE h;
    char * hptr;

    checkparam(1, 2) ;

    rxstrdup(hptr, argv[0]);
    sscanf(hptr, "%lx", (unsigned long *)&h);

    if (argc == 2 && argv[1].strptr) {
	rxstrdup(filename, argv[1]);
    }

    if (!ClearEventLog(h, filename)) {
	result_one();
    }
    else {
	result_zero();
    }

    return 0;
}

static void seteventvar(PRXSTRING stemname, EVENTLOGRECORD * elr);

rxfunc(w32findeventlogentry)
{
    HANDLE h;
    DWORD recnum;
    DWORD cbRead, needed;
    char chDummy[1];
    char * hptr;

    checkparam(2, 3) ;

    rxstrdup(hptr, argv[0]);
    sscanf(hptr, "%lx", (unsigned long *)&h);

    rxstrdup(hptr, argv[1]);
    if (sscanf(hptr, "%lu", &recnum) != 1 || recnum == 0) {
	return BADARGS;
    }

    ReadEventLog(h, EVENTLOG_SEEK_READ|EVENTLOG_FORWARDS_READ, recnum,
		 &chDummy, 0, &cbRead, &needed);

    if (GetLastError() != ERROR_INSUFFICIENT_BUFFER) {
	result_one();
    }
    else {
	/* either eLog is NULL, so realloc will act like malloc, or it isn't,
	 * so realloc will possibly do nothing */
	eLog = realloc(eLog, needed);

	if (!eLog) {
	    result_one();
	}

	else if (!ReadEventLog(h, EVENTLOG_SEEK_READ|EVENTLOG_FORWARDS_READ, recnum,
			       eLog, needed, &cbRead, &needed)) {

	    result_one();
	}
	else {
	    result_zero();
	}

	if (argc == 3)
	    seteventvar(argv+2, (EVENTLOGRECORD *)eLog);
    }

    return 0;
}

rxfunc(w32geteventid)
{
    EVENTLOGRECORD *elr = (EVENTLOGRECORD *)eLog;

    checkparam(0, 0) ;

    if (!elr) {
	rc_one();
	result->strlength = sprintf(result->strptr, "-1");
    }
    else {
	rc_zero();
	result->strlength = sprintf(result->strptr, "%lu", elr->EventID);
    }

    return 0;
}



rxfunc(w32geteventtype)
{
    EVENTLOGRECORD *elr = (EVENTLOGRECORD *)eLog;

    checkparam(0, 0) ;

    if (!elr) {
	rc_one();
	result->strlength = sprintf(result->strptr, "-1");
    }
    else {
	rc_zero();
	result->strlength = sprintf(result->strptr, "%lu", elr->EventType);
    }

    return 0;
}



rxfunc(w32geteventcategory)
{
    EVENTLOGRECORD *elr = (EVENTLOGRECORD *)eLog;

    checkparam(0, 0) ;

    if (!elr) {
	rc_one();
	result->strlength = sprintf(result->strptr, "-1");
    }
    else {
	rc_zero();
	result->strlength = sprintf(result->strptr, "%lu", elr->EventCategory);
    }

    return 0;
}



rxfunc(w32geteventnumstrings)
{
    EVENTLOGRECORD *elr = (EVENTLOGRECORD *)eLog;

    checkparam(0, 0) ;

    if (!elr) {
	rc_one();
	result->strlength = sprintf(result->strptr, "-1");
    }
    else {
	rc_zero();
	result->strlength = sprintf(result->strptr, "%lu", elr->NumStrings);
    }

    return 0;
}


/* change from ataman: ataman returned date in asctime() format, which
 * is Ddd Mmm d hh:mm:ss yyyy (ie, useless). I return time in
 * yyyy-mm-dd hh.mm.ss, which is one of the ISO formats.
 * The (enhanced) Regina  date function can be used to convert this
 * to some other format. (Actually I thought better of it, and only
 * return the ISO format in w32findeventlogentry) */
rxfunc(w32geteventtimewritten)
{
    EVENTLOGRECORD *elr = (EVENTLOGRECORD *)eLog;
    struct tm* tm;
    char * p;

    checkparam(0, 0) ;

    if (!elr) {
	rc_one();
	result->strlength = sprintf(result->strptr, "-1");
    }
    else {
	rc_zero();
	tm = localtime(&elr->TimeWritten);

#ifdef USE_REXX_DATES
	result->strlength = sprintf(result->strptr, "%04d%02d%02d %02d:%02d:%02d",
		tm->tm_year, tm->tm_mon+1, tm->tm_mday, tm->tm_hour,
		tm->tm_min, tm->tm_sec);
#else
	p = asctime(tm);
	result->strlength = strchr(p, '\n') - p;
	memcpy(result->strptr, p, result->strlength);
#endif
	}

    return 0;
}


/* see the comments to w32geteventtimewritten for the output format */
rxfunc(w32geteventtimegenerated)
{
    EVENTLOGRECORD *elr = (EVENTLOGRECORD *)eLog;
    struct tm* tm;
    char * p;

    checkparam(0, 0) ;

    if (!elr) {
	rc_one();
	result->strlength = sprintf(result->strptr, "-1");
    }
    else {
	rc_zero();
	tm = localtime(&elr->TimeGenerated);
#ifdef USE_REXX_DATES
	result->strlength = sprintf(result->strptr, "%04d%02d%02d %02d:%02d:%02d",
		tm->tm_year, tm->tm_mon+1, tm->tm_mday, tm->tm_hour,
		tm->tm_min, tm->tm_sec);
#else
	p = asctime(tm);
	result->strlength = strchr(p, '\n') - p;
	memcpy(result->strptr, p, result->strlength);
#endif
	}

    return 0;
}



rxfunc(w32geteventstring)
{
    EVENTLOGRECORD *elr = (EVENTLOGRECORD *)eLog;
    char *szbuf = NULL;
    DWORD dw;
    DWORD szwanted;
    char *p;
    char * hptr;

    checkparam(1, 1) ;

    if (!elr) {
	rc_one();
	result->strlength = 0;
    }
    else {
	rc_zero();

	rxstrdup(hptr, argv[0]);
	if (sscanf(hptr, "%lu", &szwanted) != 1) {
	    return BADARGS;
	}

	if (szwanted == 0 || szwanted > elr->NumStrings) {
	    return BADARGS;
	}

	for (p=&eLog[elr->StringOffset], dw=1; dw < szwanted; dw++) {
	    /* replaced while(*p) p++; on the theory strchr must be as fast,
	     * but it's probably faster, unless the compiler people blew it */
	    p = strchr(p, '\0');
	    if (p)
		p++;
	}

	if (p) {
	    rxresize(result, strlen(p));
	    memcpy(result->strptr, p, result->strlength);
	}
	else {
	    rc_one();
	    result->strlength = 0;
	}
    }

    return 0;
}



rxfunc(w32geteventdata)
{
    EVENTLOGRECORD *elr = (EVENTLOGRECORD *)eLog;
    DWORD dw;
    char *szbuf;
    unsigned char *p = (unsigned char *)&eLog[elr->DataOffset];

    checkparam(0, 0) ;

    szbuf = malloc((elr->DataLength*2)+1);

    if (!szbuf) {
	rc_one();
    }
    else {
	rc_zero();

	for (dw = 0; dw < elr->DataLength; dw++) {
	    sprintf(&szbuf[dw*2], "%02x", (unsigned)p[dw]);
	}

	rxresize(result, elr->DataLength * 2);
	memcpy(result->strptr, szbuf, result->strlength);

	free(szbuf);
    }

    return 0;
}


rxfunc(w32writeeventlog)
{
    WORD type;
    HANDLE h = NULL;
    char *hostname = NULL;
    char *source;
    DWORD datalen = 0;
    WORD category;
    DWORD eventid;
    WORD nStrings = 0;
    char **ppsz;
    unsigned char *pData = NULL;
    char * strptr;
    register int i;

    /* be default, assume there'll be an error */
    result_one();

    if (argc < 6 || !argv[1].strptr) {
	return BADARGS;
    }

    nStrings = (unsigned short)(argc-6);
    ppsz = alloca(nStrings * sizeof(*ppsz));

    /* set some strings */
    if (argv[0].strptr) {
	rxstrdup(hostname, argv[0]);
    }

    rxstrdup(source, argv[1]);

    if (argv[2].strptr) {
	rxstrdup(strptr, argv[2]);
	if (sscanf(strptr, "%hu", (unsigned short *)&type) != 1) {
	    goto blowup;
	}
    } else {
	type = EVENTLOG_ERROR_TYPE;
    }

    if (argv[3].strptr) {
	rxstrdup(strptr,argv[3]);
	if (sscanf(strptr, "%hu", (unsigned short *)&category) != 1) {
	    goto blowup;
	}
    } else {
	category = 0;
    }

    if (argv[4].strptr) {
	rxstrdup(strptr, argv[4]);
	if (sscanf(strptr, "%lu", (unsigned long *)&eventid) != 1) {
	    goto blowup;
	}
    } else {
	eventid = 0;
    }

    if (argv[5].strptr) {
	pData = RXSTRPTR(argv[5]);
	datalen = RXSTRLEN(argv[5]);
    }

    for (i = 6; i < argc; i++) {
	if (argv[i].strptr) {
	    rxstrdup(ppsz[i-6], argv[i]);
	} else {
	    ppsz[i-6] = "";
	}
    }

    if ((h=RegisterEventSource(hostname, source)) == NULL) {
	goto blowup;
    }

    if (!ReportEvent(h, type, category, eventid, NULL, nStrings, datalen, ppsz, pData)) {
	goto blowup;
    }

    if (!CloseEventLog(h)) {
	goto blowup;
    }

    result_zero();
    h = NULL;

blowup:

    if (h) {
	(void)CloseEventLog(h);
    }

    return 0;
}


static struct tm * correcttime(time_t thetime)
{
    struct tm * ptm;

    ptm = localtime(&thetime);

    /* if it's daylight savings time, we want to subtract an hour, and all
     * that that implies */
    if (ptm->tm_isdst) {
	thetime -= 3600;
	ptm = localtime(&thetime);
    }

    /* make it a 4 digit year */
    ptm->tm_year += 1900;

    return ptm;
}


static HMODULE GetEventMessageModule(const char * app)
{
   static const char regentry[] = "System\\CurrentControlSet\\Services\\EventLog\\%s\\%s",
      regvalue[] = "EventMessageFile";
   char evt[sizeof(regentry)+100], buf[MAX_PATH+1], buf2[MAX_PATH+1];
   DWORD bufl = sizeof(buf);
   HKEY hkey = NULL;
   HMODULE hnd = NULL;

   sprintf(evt, regentry, theLog, app);
   RegOpenKeyEx(HKEY_LOCAL_MACHINE, evt, 0, KEY_QUERY_VALUE, &hkey);
   if (hkey) {
      RegQueryValueEx(hkey, regvalue, NULL, NULL, buf2, &bufl);
      ExpandEnvironmentStrings(buf2, buf, sizeof(buf));
      hnd = LoadLibrary(buf);
      RegCloseKey(hkey);
   }
   
   return hnd;
}

static HMODULE GetCategoryMessageModule(const char * app)
{
   static const char regentry[] = "System\\CurrentControlSet\\Services\\EventLog\\%s\\%s",
      regvalue[] = "CategoryMessageFile";
   char evt[sizeof(regentry)+100], buf[MAX_PATH+1];
   DWORD bufl = sizeof(buf);
   HKEY hkey = NULL;
   HMODULE hnd = NULL;

   sprintf(evt, regentry, theLog, app);
   RegOpenKeyEx(HKEY_LOCAL_MACHINE, evt, 0, KEY_QUERY_VALUE, &hkey);
   if (hkey) {
      RegQueryValueEx(hkey, regvalue, NULL, NULL, buf, &bufl);
      hnd = LoadLibrary(buf);
      RegCloseKey(hkey);
   }
   
   return hnd;
}

/* the high few bytes of the id seem to be used for severity masks */
#define ID_MASK 0xffff

/* set up a stem variable particular to reading an event log entry.
 * This is in lieu of calling all those getthis, getthat functions */
static void seteventvar(PRXSTRING stemname, EVENTLOGRECORD * elr)
{
    SHVBLOCK *evstem, * pv;
    char intdata[7][20], * stem;
    register int i = 0;
    register char * cp;
    struct tm * ptm;
    char **args, * msgbuf;
    int msglen;
    HMODULE hnd;
    static const char info[] = "Information", err[] = "Error", warn[] = "Warning",
       saud[] = "Success Audit", faud[] = "Failure Audit", succ[] = "Success",
       none[] = "None";
    

    rxstrdup(stem, *stemname);

    evstem = alloca((12+elr->NumStrings)*sizeof(SHVBLOCK));
    memset(evstem, 0, sizeof(evstem));

    /* first drop it */
    pv = evstem;
    pv->shvnext = pv + 1;
    pv->shvname = *stemname;
    pv->shvcode = RXSHV_DROPV;

    /* now do each of the integer members */
    pv++;
    pv->shvnext = pv + 1;
    pv->shvname.strptr = alloca(stemname->strlength+3);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "ID");
    pv->shvvalue.strptr = intdata[i++];
    pv->shvvalue.strlength = sprintf(pv->shvvalue.strptr, "%lu", elr->EventID&ID_MASK);
    pv->shvcode = RXSHV_SET;

    pv++;
    pv->shvnext = pv + 1;
    pv->shvname.strptr = alloca(stemname->strlength+5);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "TYPE");

    switch (elr->EventType) {
       case EVENTLOG_SUCCESS: pv->shvvalue.strptr = succ;
          pv->shvvalue.strlength = sizeof(succ) - 1;
          break;
       case EVENTLOG_ERROR_TYPE: pv->shvvalue.strptr = err;
          pv->shvvalue.strlength = sizeof(err) - 1;
          break;
       case EVENTLOG_WARNING_TYPE: pv->shvvalue.strptr = warn;
          pv->shvvalue.strlength = sizeof(warn) - 1;
          break;
       case EVENTLOG_INFORMATION_TYPE: pv->shvvalue.strptr = info;
          pv->shvvalue.strlength = sizeof(info) - 1;
          break;
       case EVENTLOG_AUDIT_SUCCESS: pv->shvvalue.strptr = saud;
          pv->shvvalue.strlength = sizeof(saud) - 1;
          break;
       case EVENTLOG_AUDIT_FAILURE: pv->shvvalue.strptr = faud;
          pv->shvvalue.strlength = sizeof(faud) - 1;
          break;
       default:
          pv->shvvalue.strptr = intdata[i++];
          pv->shvvalue.strlength = sprintf(pv->shvvalue.strptr, "%d", elr->EventType);
    }
    pv->shvcode = RXSHV_SET;

    pv++;
    pv->shvnext = pv + 1;
    pv->shvname.strptr = alloca(stemname->strlength+9);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "CATEGORY");

    hnd = GetCategoryMessageModule((char *)(elr+1));
    if (hnd) {
       pv->shvvalue.strptr = intdata[i++];
       FormatMessage(FORMAT_MESSAGE_FROM_HMODULE,
          hnd, elr->EventCategory, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (char *)pv->shvvalue.strptr, sizeof(*intdata),
          NULL);
       CloseHandle(hnd);
       hnd = NULL;
       pv->shvvalue.strlength = strlen(pv->shvvalue.strptr);
    }
    else {
       pv->shvvalue.strptr = none;
       pv->shvvalue.strlength = sizeof(none)-1;
    }
    pv->shvcode = RXSHV_SET;

    pv++;
    pv->shvnext = pv + 1;
    pv->shvname.strptr = alloca(stemname->strlength+11);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "NUMSTRINGS");
    pv->shvvalue.strptr = intdata[i++];
    pv->shvvalue.strlength = sprintf(pv->shvvalue.strptr, "%d", elr->NumStrings);
    pv->shvcode = RXSHV_SET;

    /* this is pointless, but pseudo-conventional */
    pv++;
    pv->shvnext = pv + 1;
    pv->shvname.strptr = alloca(stemname->strlength+9);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "STRING.0");
    pv->shvvalue.strptr = intdata[i++];
    pv->shvvalue.strlength = sprintf(pv->shvvalue.strptr, "%d", elr->NumStrings);
    pv->shvcode = RXSHV_SET;

    pv++;
    pv->shvnext = pv + 1;
    pv->shvname.strptr = alloca(stemname->strlength+12);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "TIMEWRITTEN");
    pv->shvvalue.strptr = intdata[i++];
    ptm = correcttime(elr->TimeWritten);

    pv->shvvalue.strlength = sprintf(pv->shvvalue.strptr, "%04d%02d%02d %02d:%02d:%02d",
    				     ptm->tm_year, ptm->tm_mon+1, ptm->tm_mday, ptm->tm_hour,
				     ptm->tm_min, ptm->tm_sec);
    pv->shvcode = RXSHV_SET;

    pv++;
    pv->shvnext = pv + 1;
    pv->shvname.strptr = alloca(stemname->strlength+14);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "TIMEGENERATED");
    pv->shvvalue.strptr = intdata[i++];

    ptm = correcttime(elr->TimeGenerated);

    pv->shvvalue.strlength = sprintf(pv->shvvalue.strptr, "%04d%02d%02d %02d:%02d:%02d",
    				     ptm->tm_year, ptm->tm_mon+1, ptm->tm_mday, ptm->tm_hour,
				     ptm->tm_min, ptm->tm_sec);
    pv->shvcode = RXSHV_SET;

    /* the service name (some messages depend on knowing this */
    pv++;
    pv->shvnext = pv + 1;
    pv->shvname.strptr = alloca(stemname->strlength+8);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "SOURCE");
    pv->shvvalue.strptr = (char *)(elr+1);
    pv->shvvalue.strlength = strlen(pv->shvvalue.strptr);
    pv->shvcode = RXSHV_SET;

    /* now the data buffer */
    pv++;
    pv->shvnext = pv + 1;
    pv->shvname.strptr = alloca(stemname->strlength+5);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "DATA");
    /* note -- we don't put this in hex. */
    pv->shvvalue.strptr = ((char *)elr) + elr->DataOffset;
    pv->shvvalue.strlength = elr->DataLength;
    pv->shvcode = RXSHV_SET;

    /* now do the strings */
    if (elr->NumStrings)
       args = alloca(sizeof(*args)*elr->NumStrings);

    cp = ((char *)elr)+elr->StringOffset;
    for (i = 1; i <= elr->NumStrings; i++) {
	pv++;
	pv->shvnext = pv + 1;
	pv->shvname.strptr = alloca(stemname->strlength+15);
	pv->shvname.strlength = sprintf(pv->shvname.strptr, "%sSTRING.%d", stem, i);
	pv->shvvalue.strptr = cp;
	pv->shvvalue.strlength = strlen(cp);
	pv->shvcode = RXSHV_SET;
        args[i-1] = cp;
	cp = strchr(cp, '\0') + 1;
    }


    hnd = GetEventMessageModule((char *)(elr+1));
    FormatMessage(FORMAT_MESSAGE_ARGUMENT_ARRAY|FORMAT_MESSAGE_FROM_HMODULE|FORMAT_MESSAGE_ALLOCATE_BUFFER,
       hnd, elr->EventID, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (char *)&msgbuf, 0,
       args);
    if (hnd)
       CloseHandle(hnd);

    pv++;
    pv->shvnext = NULL;
    pv->shvname.strptr = alloca(stemname->strlength+12);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "DESCRIPTION");
    if (msgbuf) {
       pv->shvvalue.strptr = msgbuf;
       pv->shvvalue.strlength = strlen(msgbuf);
    }
    else {
       pv->shvvalue.strlength = 0;
    }
    pv->shvcode = RXSHV_SET;

    RexxVariablePool(evstem);

    if (msgbuf)
       LocalFree(msgbuf);
}


/* mapping between rexx strings and C symbolic names for registry data types
 * given that there are only 9 types, I'm just using a straight scan
 * of this array, although I sorted the entries, so you could use bsearch
 * if you wanted to.
 */
static struct type_map_s {
    char * name;
    DWORD type;
} theTypeMap[] = {
    "REG_BINARY", REG_BINARY,
    "REG_DWORD", REG_DWORD,
    "REG_DWORD_BIG_ENDIAN", REG_DWORD_BIG_ENDIAN,
    "REG_EXPAND_SZ", REG_EXPAND_SZ,
    "REG_LINK", REG_LINK,
    "REG_MULTI_SZ", REG_MULTI_SZ,
    "REG_NONE", REG_NONE,
    "REG_RESOURCE_LIST", REG_RESOURCE_LIST,
    "REG_SZ", REG_SZ
};

static char * type_name(DWORD dwType)
{
    register int i;

    for (i = 0; i < DIM(theTypeMap); i++) {
	if (dwType == theTypeMap[i].type)
	    return theTypeMap[i].name;
    }

    return "UNKNOWN";
 }

static DWORD type_type(const char * const theType)
{
    register int i;

    for (i = 0; i < DIM(theTypeMap); i++) {
	if (!stricmp(theType, theTypeMap[i].name))
	    return theTypeMap[i].type;
    }

    return 0;
}


/* report the version and location of dll */ 
rxfunc(w32version)
{
   char buf[1024];
   int bufl = 6;

   memcpy(buf, "1.2.5 ", 6);
   bufl += GetModuleFileName(GetModuleHandle("W32UTIL.DLL"), buf+bufl, sizeof(buf)-bufl);

   rxresize(result, bufl);
   memcpy(result->strptr, buf, bufl);

   return 0;
}

/* load and unload all the functions */

static const struct {
   char * name;
   APIRET (APIENTRY*rxFcn)(PUCHAR fname, ULONG argc, PRXSTRING argv, PSZ pSomething, PRXSTRING result);
} funcTable[] = {
   "w32backupeventlog", w32backupeventlog,
   "w32callfunc", w32callfunc,
   "w32callproc", w32callproc,
   "w32cleareventlog", w32cleareventlog,
   "w32closeeventlog", w32closeeventlog,
   "w32createobject", w32createobject,
   "w32findeventlogentry", w32findeventlogentry,
   "w32geterror", w32geterror,
   "w32geteventcategory", w32geteventcategory,
   "w32geteventdata", w32geteventdata,
   "w32geteventid", w32geteventid,
   "w32geteventnumstrings", w32geteventnumstrings,
   "w32geteventstring", w32geteventstring,
   "w32geteventtimegenerated", w32geteventtimegenerated,
   "w32geteventtimewritten", w32geteventtimewritten,
   "w32geteventtype", w32geteventtype,
   "w32getnumberofeventlogrecords", w32getnumberofeventlogrecords,
   "w32getoldesteventlogrecord", w32getoldesteventlogrecord,
   "w32getobject", w32getobject,
   "w32getproperty", w32getproperty,
   "w32getsubobj", w32getsubobj,
   "w32olegeterror", w32olegeterror,
   "w32olenext", w32olenext,
   "w32openbackupeventlog", w32openbackupeventlog,
   "w32openeventlog", w32openeventlog,
   "w32putproperty", w32putproperty,
   "w32regclosekey", w32regclosekey,
   "w32regconnectregistry", w32regconnectregistry,
   "w32regcreatekey", w32regcreatekey,
   "w32regdeletekey", w32regdeletekey,
   "w32regdeletevalue", w32regdeletevalue,
   "w32regenumkey", w32regenumkey,
   "w32regenumvalue", w32regenumvalue,
   "w32regflushkey", w32regflushkey,
   "w32reggetkeysecdesc", w32reggetkeysecdesc,
   "w32regloadkey", w32regloadkey,
   "w32regopenkey", w32regopenkey,
   "w32regqueryinfokey", w32regqueryinfokey,
   "w32regqueryvalue", w32regqueryvalue,
   "w32regqueryvaluetype", w32regqueryvaluetype,
   "w32regrestorekey", w32regrestorekey,
   "w32regsavekey", w32regsavekey,
   "w32regsetkeysecdesc", w32regsetkeysecdesc,
   "w32regsetvalue", w32regsetvalue,
   "w32regunloadkey", w32regunloadkey,
   "w32releaseobject", w32releaseobject,
   "w32writeeventlog", w32writeeventlog,
   "w32svcstart", w32svcstart,
   "w32svcstop", w32svcstop,
   "w32svcinstall", w32svcinstall,
   "w32svcremove", w32svcremove,
   "w32menuadditem", w32menuadditem,
   "w32menumoveitem", w32menumoveitem,
   "w32menumove", w32menumove,
   "w32menuremoveitem", w32menuremoveitem,
   "w32version", w32version,
   "w32dropfuncs", w32dropfuncs,
   "w32loadfuncs", w32loadfuncs        /* must be last, so we don't reload it */
};

/* load every function in that table. */
rxfunc(w32loadfuncs)
{
   register int i;

   /* can use RegisterFunctionExe since the addresses are already resolved in
    * the array above */
   for (i = 0; i < DIM(funcTable)-1; i++) {
      RexxRegisterFunctionExe(funcTable[i].name, funcTable[i].rxFcn);
      /* ought to test for an error */
   }

   result_zero();

   return 0;
}

/* unload every function in that table */
rxfunc(w32dropfuncs)
{
   register int i;

   for (i = 0; i < DIM(funcTable); i++) {
      RexxDeregisterFunction(funcTable[i].name);
      /* ought to test for an error */
   }

   result_zero();

   return 0;
}

/* copy the contents of a system error for the most recent error message into the
 * errmsg buffer */

rxfunc(w32geterror)
{
   int lasterror = GetLastError();

   result->strlength = sprintf(result->strptr, "%d: ", lasterror);

   result->strlength += FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, NULL, lasterror,
                 MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (char *)result->strptr+result->strlength,
                 DEFAULTSTRINGSIZE - result->strlength, NULL);
   return 0;
}
