/* service control manager functions:
 *  Copyright 1998 Patrick TJ McPhee. All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Header: C:/ptjm/rexx/w32funcs/RCS/services.c 1.3 1999/12/29 22:20:02 pmcphee Rel $
 */

/* functions defined in this file:
 * w32svcstart(servicename)
 * w32svcstop(servicename)
 * w32svcremove(servicename)
 * w32svcinstall(servicename, displayname, programpath[, autostart, user, password])
 * all functions return 0 for success, or an error code. Note: start and stop return 1
 * to mean the function was called unneccesarily.
 * for start, stop, and remove, servicename can be either the registry key for
 * the service, or the display name as shown in the service control manager
 */ 
 

#include <windows.h>
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <direct.h>
#include "rxproto.h"

static SC_HANDLE smHandle = NULL;

static int svcconnect()
{
   if (!smHandle) {
      smHandle = OpenSCManager(0, 0, SC_MANAGER_ALL_ACCESS);
   }

   return smHandle != NULL;
}

static void svcdisconnect()
{
   if (smHandle != NULL)
      CloseServiceHandle(smHandle);
   smHandle = NULL;
}



int Install(const char * const servName, const char * const dispName,
                          const char * const path, const char * const user,
                          const char * const pwd, BOOL autoStart)
{
   SC_HANDLE srv;

   if (!svcconnect())
      return -1;

   srv = CreateService(smHandle, servName, dispName, SERVICE_ALL_ACCESS,
                       SERVICE_WIN32_OWN_PROCESS, autoStart ? SERVICE_AUTO_START : SERVICE_DEMAND_START,
                       SERVICE_ERROR_NORMAL, path, 0, 0, 0, user, pwd);

   if (srv == NULL) {
      return -2;
   }


   CloseServiceHandle(srv);
   return 0;
}

int Remove(const char * const servName)
{
   SC_HANDLE      srv;
   SERVICE_STATUS stat;
   int rc = 0;
   char key[100];
   long len = sizeof(key);

   if (!svcconnect())
      return -1;
   
   srv = OpenService(smHandle, servName, SERVICE_ALL_ACCESS | DELETE);

   if (!srv) {
      if (GetServiceKeyName(smHandle, servName, key, &len)) {
         srv = OpenService(smHandle, key, SERVICE_ALL_ACCESS | DELETE);
      }
   }

   if (!srv)
      return -2;

   if (!QueryServiceStatus(srv, &stat))
      rc = -3;

   else {
      if (stat.dwCurrentState != SERVICE_STOPPED) {
         if (!ControlService(srv, SERVICE_CONTROL_STOP, &stat))
            return -4;
      }

      if (!DeleteService(srv))
         rc = -5;
   }

   CloseServiceHandle(srv);

   return rc;
}

int Start(const char * const servName)
{
   SC_HANDLE srv;
   SERVICE_STATUS stat;
   int rc = 0;
   char key[100];
   long len = sizeof(key);

   if (!svcconnect())
      return -1;
   
   srv = OpenService(smHandle, servName, SERVICE_QUERY_STATUS|SERVICE_START);

   if (!srv) {
      if (GetServiceKeyName(smHandle, servName, key, &len)) {
         srv = OpenService(smHandle, key, SERVICE_QUERY_STATUS|SERVICE_START);
      }
   }

   if (!srv)
      return -2;

   if (!QueryServiceStatus(srv, &stat)) {
      rc = -3;
   }

   else if (stat.dwCurrentState != SERVICE_STOPPED) {
      rc = 1;   /* not really an error */
   }

   else if (!StartService(srv, 0, NULL)) {
      rc = -4;
   }

   CloseServiceHandle(srv);
   return rc;
}

int Stop(const char * const servName)
{
   int rc = 0;
   SC_HANDLE srv;
   SERVICE_STATUS stat;
   char key[100];
   long len = sizeof(key);

   if (!svcconnect())
      return -1;
   
   srv = OpenService(smHandle, servName, SERVICE_QUERY_STATUS|SERVICE_STOP);

   if (!srv) {
      if (GetServiceKeyName(smHandle, servName, key, &len)) {
         srv = OpenService(smHandle, key, SERVICE_QUERY_STATUS|SERVICE_STOP);
      }
   }

   if (srv == NULL)
      return -2;

   if (!QueryServiceStatus(srv, &stat)) {
      rc = -3;
   }

   else if (stat.dwCurrentState == SERVICE_STOPPED) {
      rc = 1;   /* not really an error */
   }

   if (!ControlService(srv, SERVICE_CONTROL_STOP, &stat))
      return -4;

   CloseServiceHandle(srv);

   return rc;
}

/* w32svcstart servicename */
rxfunc(w32svcstart)
{
   char * servName;
   int rc;

   checkparam(1,1);
   rxstrdup(servName, argv[0]);
   rc = Start(servName);
   result->strlength = sprintf(result->strptr, "%d", rc);

   return 0;
}

/* svcstop servicename */
rxfunc(w32svcstop)
{
   char * servName;
   int rc;

   checkparam(1,1);
   rxstrdup(servName, argv[0]);
   rc = Stop(servName);
   result->strlength = sprintf(result->strptr, "%d", rc);

   return 0;
}

/* svcremove servicename */
rxfunc(w32svcremove)
{
   char * servName;
   int rc;

   checkparam(1,1);
   rxstrdup(servName, argv[0]);
   rc = Remove(servName);
   result->strlength = sprintf(result->strptr, "%d", rc);

   return 0;
}

/* svcinstall servicename, display name, program path[, autostart, user, password] */
rxfunc(w32svcinstall)
{
   char * servName, *dispname, *path, *as = "0", *user = NULL, *pwd = NULL;
   int rc;

   checkparam(3,6);
   rxstrdup(servName, argv[0]);
   rxstrdup(dispname, argv[1]);
   rxstrdup(path, argv[2]);

   if (argc > 3 && argv[3].strlength)
      rxstrdup(as, argv[3]);

   if (argc > 4 && argv[4].strlength)
      rxstrdup(user, argv[4]);

   if (argc > 5 && argv[5].strlength)
      rxstrdup(pwd, argv[5]);

   rc = Install(servName, dispname, path, user, pwd, atoi(as));
   result->strlength = sprintf(result->strptr, "%d", rc);

   return 0;
}
